function raster(varargin)
% raster takes in (data, format) and renders a raster plot
% of the data.  It accepts 2 forms for the data: 
%   "times" - the data matrix contains spike times for each trial
%   "spikes" - the data matrix contains actual spikes for each time bin (1s and 0s)
% In the data matrix, the rows are always trials.

data = varargin{1};
if (nargin == 2)
    format = varargin{2};
else 
    format = 'times';
end

if (strcmp(format,'spikes') == 1)
    [I J] = find(data);
    n = hist(I, size(data,1));
    numcols = max(n);
    data2 = zeros(size(data,1), numcols);
    for i=1:size(data,1)
        tmp = find(data(i,:));
        data2(i,:) = cat(2, tmp, zeros(1, numcols - length(tmp)));
    end
    data = data2;
end

figure
hold on

lightc = [98 0 255] ./ 255;  % Color of light bar
lights = [205 179 255] ./ 255;  % Color of light shading
patch([10 10 20 20], [0 size(data,1)+2 size(data,1)+2 0], lights, 'LineStyle', 'none'); %The first arguments of patch position the vertices of a polygon
text(15, size(data,1)+1.8, 'Light', 'Color', 'black', ...
     'FontSize', 10, 'HorizontalAlignment', 'center', 'VerticalAlignment', 'top');

%plot([10 20], [size(data,1)+1.2 size(data,1)+1.2], 'LineWidth', 6, 'Color', lightc);

if (iscell(data))
    maxLen = 0;
    for i=1:length(data)
        if (length(data{i}) > maxLen)
            maxLen = length(data{i});
        end
    end
    tmpData = zeros(length(data), maxLen);
    for i=1:length(data)
        tmpData(i,1:length(data{i})) = data{i};
    end
    data = tmpData;
end    
for i=1:size(data,1)  % For each trial
    for j=1:size(data,2)  % For each spike time
        if (data(i,j) ~= 0)
            plot([data(i,j)/1000 data(i,j)/1000], [i+0.4 i-0.4], 'k', 'LineWidth', 2);
        end
    end
end

prettify();
xlabel('time (s)', 'FontSize', 12, 'FontWeight', 'Bold', 'FontName', 'Arial');
ylabel('trial', 'FontSize', 12, 'FontWeight', 'Bold', 'FontName', 'Arial');
set(gca, 'FontName', 'Arial', 'FontSize', 12, 'YTick', [0:5:size(data,1)]);
title('Pumps over time', 'FontName', 'Arial', 'FontWeight', 'Bold', 'FontSize', 12);
set(gca, 'XTick', 1:maxLen);
xticklabels = cell(maxLen,1);
for i = 1:maxLen
    if (mod(i,5) == 0)
        xticklabels{i} = eval('num2str(i)');
    else
        xticklabels{i} = ' ';
    end
end
set(gca, 'XTickLabel', xticklabels);
ylim([0 size(data,1)+2]);